/* ###########################################################################
#
# Copyright (c) 2003-2009 Telchemy, Incorporated. All Rights Reserved.
#
# Telchemy Confidential and Proprietary
#
# The following software source code ("Software") is strictly confidential
# and is proprietary to Telchemy, Incorporated ("Telchemy").  It may only
# be read, used, copied, adapted, modified or otherwise utilized by parties
# (individuals, corporations, or organizations) that have entered into a
# license agreement or confidentiality agreement with Telchemy, and are thus
# subject to the terms of that license agreement or confidentiality agreement
# and any other applicable agreement between the party and Telchemy.  If
# there is any doubt as to whether a party is entitled to access, read, use,
# copy, adapt, modify or otherwise utilize the Software, or whether a party
# is entitled to disclose the Software to any other party, you should contact
# Telchemy.  If you, as a party, have not entered into a license agreement or
# confidentiality agreement with Telchemy granting access to this Software,
# all media, copies and printed listings containing the Software should be
# forthwith returned to Telchemy.
#
# Telchemy reserves the right to take legal action against any party that
# violates Telchemy's rights to the Software, including without limitation a
# party's breach of the above conditions.
#
# If you have questions about any of the above conditions, or wish to report
# violations, please contact:  support@telchemy.com
#
# ##########################################################################*/

/* mp2tsim.c
 *
 * VQmon/HD Sample Host Application - Stream Simulator
 *
 * Copyright (c) 2003-2009 Telchemy, Incorporated
 */

#include <stdio.h>
#include <stdlib.h>

#include "mp2tsim.h"

#include "vqmon.h"

/*
 * This application demonstrates the usage of VQmon in monitoring an IPTV
 * stream.  Artificial MPEG-2 transport stream data is indicated to VQmon
 * using VQmonInterfaceIndicatePacket, and VQmon uses the stream management
 * callback set in VQmonInitialize to indicate stream-related events to the
 * application.
 */

int main(void)
{
    vqmon_initparams_t _tInitParams;
    vqmon_handle_t _hInterface;
    vqmon_time_t _tTimestamp;
    vqmon_pktdesc_t _tPktDesc;
    vqmon_result_t _result;
    tcmyU16 _nLoop;

    vqmon_interfacestats_pkt_t _tPktStats;
    tcmyU32 _nSize;

    /*
     * Initialize the VQmon/HD Library
     */
    VQMON_MEMCLEAR(&_tInitParams, sizeof(vqmon_initparams_t));
    _tInitParams.nVersion = VQMON_IFVERSION_CURRENT;
    _tInitParams.pfStreamNotifyHdlr = MP2TSim_NotifyHandler;
    _result = VQmonInitialize(&_tInitParams);
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonInitialize failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    /*
     * Create an interface handle
     */
    _result = VQmonInterfaceCreate(&_hInterface, NULL);
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonInterfaceCreate failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    /*
     * Create video packets and indicate to VQmon
     */
    _tTimestamp.tv_sec = 0;
    _tTimestamp.tv_usec = 0;
    for (_nLoop = 0; _nLoop < 1000; _nLoop++)
    {
        MP2TSim_CreatePacket(&_tPktDesc, &_tTimestamp);

        VQmonInterfaceIndicatePacket(
                _hInterface,
                _tTimestamp,
                &_tPktDesc
                );
    }

    /*
     * End a reporting interval.
     * This will produce vqmonStreamMgmtNotifyIntervalUpdate callbacks,
     * and MP2TSim_NotifyHandler will display the interval statistics
     */
    printf("--- Ending a reporting interval\n");
    VQmonInterfaceIndicateEvent(
            _hInterface,
            vqmonInterfaceEventIntervalEnd,
            _tTimestamp
            );

    /*
     * Generate some more packets
     */
    for (_nLoop = 0; _nLoop < 1000; _nLoop++)
    {
        MP2TSim_CreatePacket(&_tPktDesc, &_tTimestamp);

        VQmonInterfaceIndicatePacket(
                _hInterface,
                _tTimestamp,
                &_tPktDesc
                );
    }

    /*
     * Terminate all streams.
     * This will produce vqmonStreamMgmtNotifyTerminated callbacks, and
     * MP2TSim_NotifyHandler will display metrics associated with the
     * stream and destroy the stream handle.
     */
    printf("--- Terminating streams\n");
    _result = VQmonInterfaceStreamsCleanupInactive(
            _hInterface,
            _tTimestamp,
            0
            );

    /*
     * Retrieve and display aggregate statistics for the interface
     */
    _nSize = sizeof(vqmon_interfacestats_pkt_t);
    VQMON_MEMCLEAR(&_tPktStats, _nSize);
    _result = VQmonInterfaceStatisticsGet(
            _hInterface,
            VQMON_IFMETRICBLOCKID_PACKETSTATS,
            &_nSize,
            &_tPktStats
            );

    printf("%d packets received on interface\n", (int) _tPktStats.nPktsRcvd);

    /*
     * Cleanup
     */
    VQmonInterfaceDestroy(_hInterface);
    VQmonCleanup();

    return 0;
}

vqmon_result_t
MP2TSim_NotifyHandler(
    const vqmon_handle_t hInterface,
    const vqmon_handle_t hStream,
    const vqmon_streammgmtnotifytype_t eNotification,
    vqmon_streammgmtnotify_info_t *pNotifyInfo
    )
{
    switch (eNotification)
    {
        case vqmonStreamMgmtNotifyCreated:
        {
            printf("MP2TSim_NotifyHandler: Stream created, handle %p\n",
                    hStream);
            break;
        }
        case vqmonStreamMgmtNotifyActivating:
        {
            printf("MP2TSim_NotifyHandler: Stream activating, handle %p\n",
                    hStream);
            break;
        }
        case vqmonStreamMgmtNotifyTerminated:
        {
            /* Display metrics for the stream */
            switch (pNotifyInfo->eStreamType)
            {
                case vqmonStreamTypeVideo:
                {
                    MP2TSim_VideoSummaryDisplay(hStream);
                    break;
                }
                default:
                {
                    printf("MP2TSim_NotifyHandler: Unsupported stream type terminated: %p\n",
                            hStream);
                    break;
                }
            }
            /* Free the memory used by the stream */
            VQmonStreamDestroy(hStream);
            break;
        }
        case vqmonStreamMgmtNotifyIntervalUpdate:
        {
            switch (pNotifyInfo->eStreamType)
            {
                case vqmonStreamTypeVideo:
                {
                    MP2TSim_VideoIntervalDisplay(hStream);
                    break;
                }
                default:
                {
                    printf("MP2TSim_NotifyHandler: Unsupported stream type interval: %p\n",
                            hStream);
                    break;
                }
            }
            break;
        }
        default:
        {
            break;
        }
    }
    return VQMON_ESUCCESS;
}

void
MP2TSim_VideoSummaryDisplay(
    vqmon_handle_t const hStream
    )
{
    tcmyU32 _nSize;
    vqmon_result_t _result;

    vqmon_streamcfg_transprotoext_t _tTransProto;
    vqmon_streammetrics_pkttrans_t _tTransMetrics;
    vqmon_streammetrics_videoqual_t _tVideoQual;

    _nSize = sizeof(vqmon_streamcfg_transprotoext_t);
    VQMON_MEMCLEAR(&_tTransProto, _nSize);
    _tTransProto.eTransportProto = vqmonStreamTransProtoMP2TS;
    _result = VQmonStreamConfigGet(
            hStream,
            VQMON_STREAMCFGBLOCKID_TRANSPORTINFO,
            &_nSize,
            &_tTransProto
            );

    _nSize = sizeof(vqmon_streammetrics_pkttrans_t);
    VQMON_MEMCLEAR(&_tTransMetrics, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_PKTTRANS,
            &_nSize,
            &_tTransMetrics
            );

    _nSize = sizeof(vqmon_streammetrics_videoqual_t);
    VQMON_MEMCLEAR(&_tVideoQual, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_VIDEOQUAL,
            &_nSize,
            &_tVideoQual
            );

    printf("Video stream %p PID %d terminating, %d packets received\n", 
            hStream,
            _tTransProto.proto.mp2ts.nPID,
            _tTransMetrics.nPktsRcvd);
    printf("Rel. MOS-V (min/avg/max): %3.2f/%3.2f/%3.2f\n",
            _tVideoQual.nMinRelativeMOS_V / 256.0,
            _tVideoQual.nAvgRelativeMOS_V / 256.0,
            _tVideoQual.nMaxRelativeMOS_V / 256.0
          );
}

void
MP2TSim_VideoIntervalDisplay(
    vqmon_handle_t const hStream
    )
{
    tcmyU32 _nSize;
    vqmon_result_t _result;

    vqmon_streamcfg_transprotoext_t _tTransProto;
    vqmon_streammetrics_videointerval_t _tVidItvl;

    _nSize = sizeof(vqmon_streamcfg_transprotoext_t);
    VQMON_MEMCLEAR(&_tTransProto, _nSize);
    _tTransProto.eTransportProto = vqmonStreamTransProtoMP2TS;
    _result = VQmonStreamConfigGet(
            hStream,
            VQMON_STREAMCFGBLOCKID_TRANSPORTINFO,
            &_nSize,
            &_tTransProto
            );

    _nSize = sizeof(vqmon_streammetrics_videointerval_t);
    VQMON_MEMCLEAR(&_tVidItvl, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_VIDEOINTERVAL,
            &_nSize,
            &_tVidItvl
            );
    printf("Video stream %p PID %d interval update\n", 
            hStream,
            _tTransProto.proto.mp2ts.nPID);
    printf("Avg. Rel. Mos-V: %f\n",
            _tVidItvl.nRelativeMOS_V / 256.0);
}

void
MP2TSim_CreatePacket(
    vqmon_pktdesc_t *pPktDesc,
    vqmon_time_t *pTimestamp
    )
{

    static tcmyU8 _pPAT[188] = {
        0x47,0x40,0x00,0xF0,0xA6,0x00,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0x00,0x00,0xB0,0x0D,0x20,
        0x3E,0xDD,0x00,0x00,0x00,0x01,0xE0,0x42,0x3B,0x39,0xF1,0xCE
    };

    static tcmyU8 _pPMT[188] = {
        0x47,0x40,0x42,0xF0,0x90,0x00,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,
        0xFF,0xFF,0xFF,0xFF,0xFF,0x00,0x02,0xB0,0x23,0x00,0x01,0xD5,0x00,0x00,0xE0,0x47,
        0xF0,0x00,0x02,0xE0,0x47,0xF0,0x00,0x81,0xE0,0x48,0xF0,0x0C,0x05,0x04,0x41,0x43,
        0x2D,0x33,0x0A,0x04,0x65,0x6E,0x67,0x00,0x37,0xFA,0x91,0x6E
    };

    static tcmyU8 _pVidStart[188] = {
        0x47,0x40,0x47,0xD0,0x00,0x00,0x01,0xE0,0x02,0x92,0xB0,0x80,0x05,0x25,0x71,0xBB,
        0xDC,0x35,0xFB,0x83,0xC5,0x2D,0x6C,0x1B,0xC2,0x1D,0x15,0x7C,0x48,0xF7,0x3D,0xD1,
        0x48,0x28,0xC3,0xD7,0xC6,0xBE,0xB5,0x33,0x6A,0x22,0xD6,0xA1,0x4F,0xFB,0x9E,0xED,
        0x17,0xB8,0x7B,0xAB,0x94,0x12,0xF2,0x2B,0x4D,0xDE,0xAB,0x7E,0xA2,0x91,0x91,0x3E,
        0x4F,0x56,0xE0,0x3C,0x2A,0xF1,0xB3,0xF9,0x69,0xF8,0x3C,0x5D,0x7C,0x82,0x23,0x9E,
        0x82,0xD6,0xB8,0x85,0x8F,0x9A,0x30,0x33,0x44,0x95,0xF6,0x96,0xCA,0xC3,0x39,0x65,
        0xCE,0xB9,0xE2,0xB9,0x9A,0x77,0x63,0xA3,0x22,0x38,0x47,0xCF,0xB7,0xC8,0xA9,0x98,
        0x71,0x20,0x8E,0x27,0xC4,0x08,0x53,0x9B,0x14,0xD0,0x74,0xB5,0x22,0xE9,0xD6,0x95,
        0x6F,0x21,0xA3,0x6B,0xA2,0xDC,0xF9,0x34,0x54,0x98,0xEB,0x93,0x84,0x34,0x11,0x58,
        0xF5,0xFF,0x3B,0xFD,0xAB,0x6A,0xCE,0xBF,0x30,0xEF,0x81,0x40,0x67,0x38,0x1B,0x4D,
        0x64,0x00,0x47,0xE8,0xA6,0xEB,0x68,0x87,0x2B,0xEB,0xF7,0x31,0xAE,0x90,0x4C,0x25,
        0xC6,0x42,0x92,0xD6,0x3A,0x2E,0xBB,0xB9,0x64,0x1A,0xB8,0x31
    };

    static tcmyU8 _pVid[188] = {
        0x47,0x00,0x47,0xD6,0xEA,0xB0,0x44,0x2D,0x6D,0xCD,0x37,0xF4,0xE6,0x13,0xCC,0xC9,
        0x0F,0x82,0x70,0x19,0xA0,0x54,0xC1,0xBE,0xB4,0x33,0x73,0x40,0xB2,0x8A,0xB8,0x29,
        0xAE,0x64,0x96,0x1C,0xDB,0x9A,0x9F,0xA1,0x1B,0x61,0x08,0x95,0xC6,0x01,0x17,0xAC,
        0xF0,0x32,0xAF,0x57,0xE4,0x2F,0x56,0xF3,0xD7,0xF8,0x95,0xC6,0xA9,0xFC,0x2A,0x35,
        0xC3,0x29,0x72,0x5E,0x68,0xA3,0x6A,0xBA,0xCA,0x1F,0xA1,0x86,0xFA,0xAE,0xC5,0x66,
        0x11,0x79,0x9B,0xE0,0x29,0x01,0xED,0x5C,0x9D,0x14,0x21,0x08,0xF3,0xC6,0xCA,0x80,
        0x68,0x7A,0x0A,0xE9,0x88,0xCC,0x43,0x3C,0xBA,0x8A,0x05,0x11,0x73,0xA7,0x57,0x30,
        0x9E,0x42,0x35,0xF2,0x67,0x64,0xAD,0xA3,0x3C,0x88,0x5C,0xB2,0xBA,0xC0,0xF6,0xAE,
        0x85,0x42,0x5F,0x90,0x56,0x49,0xFD,0x83,0xCD,0xB1,0x41,0x60,0x61,0xB8,0x4B,0x5A,
        0x7C,0x83,0x11,0xDA,0xD8,0x46,0x8F,0xAE,0x2B,0x7F,0x57,0xD2,0xAD,0xCD,0xD0,0x3D,
        0x2F,0x77,0x15,0x42,0x7A,0x8A,0x3A,0x2E,0xBB,0xC6,0xE3,0x68,0x34,0x50,0x94,0x23,
        0x55,0xDA,0x9B,0x69,0x7A,0xC4,0x5A,0x0B,0x73,0xC2,0x98,0x06
    };

    static tcmyU8 _pPATCC = 0;
    static tcmyU8 _pPMTCC = 0;
    static tcmyU8 _pVidCC = 0;

    static tcmyU8 _pPktHeader[28] = {
        0x45,0x00,0x05,0x40,0x40,0x34,0x40,0x00,0x01,0x11,0x78,0xF2,0xC0,0xA8,0x09,0xDC,
        0xEF,0xFF,0x01,0x03,0x80,0x03,0x07,0xD0,0x05,0x2C,0xE4,0x35
    };

    static tcmyU8 _nState = 0;
    static tcmyU8 _pPktBuf[1344];
    static tcmyU16 _nIFramePkts = 0;
    static tcmyU16 _nPFramePkts = 0;
    static tcmyU16 _nPFrames = 0;

    tcmyU32 _nLoop = 0;

    VQMON_MEMCLEAR(_pPktBuf, sizeof(_pPktBuf));
    /*
     * Add IP and UDP headers
     */
    VQMON_MEMCOPY(_pPktBuf, _pPktHeader, sizeof(_pPktHeader));

    /*
     * Add 7 MP2T packets
     */
    for (_nLoop = 0; _nLoop < 7; _nLoop++)
    {
        switch (_nState)
        {
            case 0:
            {
                /*
                 * Send a PAT
                 */
                _pPAT[3] = (_pPAT[3] & 0xF0) | _pPATCC;
                _pPATCC = ((_pPATCC + 1) & 0x0F);
                VQMON_MEMCOPY((_pPktBuf + sizeof(_pPktHeader)), _pPAT, 188);
                _nState = 1;
                break;
            }

            case 1:
            {
                /*
                 * Send a PMT
                 */
                _pPMT[3] = (_pPMT[3] & 0xF0) | _pPMTCC;
                _pPMTCC = ((_pPMTCC + 1) & 0x0F);
                VQMON_MEMCOPY((_pPktBuf + sizeof(_pPktHeader) + (188 * _nLoop)),
                        _pPMT, 188);
                _nState = 2;
                break;
            }

            case 2:
            {
                /*
                 * Send an I-frame
                 */
                _pVidStart[3] = (_pVidStart[3] & 0xF0) | _pVidCC;
                _pVidCC = ((_pVidCC + 1) & 0x0F);
                VQMON_MEMCOPY((_pPktBuf + sizeof(_pPktHeader) + (188 * _nLoop)),
                        _pVidStart, 188);

                _nState = 3;
                break;
            }

            case 3:
            {
                /*
                 * I-Frame data
                 */

                _pVid[3] = (_pVid[3] & 0xF0) | _pVidCC;
                _pVidCC = ((_pVidCC + 1) & 0x0F);
                VQMON_MEMCOPY((_pPktBuf + sizeof(_pPktHeader) + (188 * _nLoop)),
                            _pVid, 188);
                _nIFramePkts++;

                if (_nIFramePkts >= 100)
                {
                    _nIFramePkts++;
                    _nState = 4;
                }
                break;
            }

            case 4:
            {
                /*
                 * Send a P-frame
                 */
                _pVidStart[3] = (_pVidStart[3] & 0xF0) | _pVidCC;
                _pVidCC = ((_pVidCC + 1) & 0x0F);
                VQMON_MEMCOPY((_pPktBuf + sizeof(_pPktHeader) + (188 * _nLoop)),
                        _pVidStart, 188);
                _nState = 5;
                break;
            }

            case 5:
            {
                /*
                 * P-frame data
                 */
                _pVid[3] = (_pVid[3] & 0xF0) | _pVidCC;
                _pVidCC = ((_pVidCC + 1) & 0x0F);
                VQMON_MEMCOPY((_pPktBuf + sizeof(_pPktHeader) + (188 * _nLoop)),
                            _pVid, 188);
                _nPFramePkts++;

                if (_nPFramePkts >= 15)
                {
                    _nPFrames++;
                    if (_nPFrames >= 29)
                    {
                        _nPFramePkts = 0;
                        _nPFrames = 0;
                        _nState = 0;
                    }
                    else
                    {
                        _nPFramePkts = 0;
                        _nState = 4;
                    }
                }

                break;
            }

            default:
            {
                _nState = 0;
                break;
            }
        }
    }
                

    VQMON_MEMCLEAR(pPktDesc, sizeof(vqmon_pktdesc_t));
    pPktDesc->ePktStartIndicator = vqmonPktDescStartIPHeader;
    pPktDesc->nCapturedPktLen = sizeof(_pPktBuf);
    pPktDesc->nTotalPktLen = sizeof(_pPktBuf);
    pPktDesc->pPktBuffer = _pPktBuf;

    if (_nState == 3)
    {
        pTimestamp->tv_usec += 2143;
    }
    else
    {
        pTimestamp->tv_usec += 1875;
    }

    if (pTimestamp->tv_usec >= 1000000)
    {
        pTimestamp->tv_sec++;
        pTimestamp->tv_usec -= 1000000;
    }
}
