/* ###########################################################################
#
# Copyright (c) 2011-2012 Telchemy, Incorporated. All Rights Reserved.
#
# Telchemy Confidential and Proprietary
#
# The following software source code ("Software") is strictly confidential
# and is proprietary to Telchemy, Incorporated ("Telchemy").  It may only
# be read, used, copied, adapted, modified or otherwise utilized by parties
# (individuals, corporations, or organizations) that have entered into a
# license agreement or confidentiality agreement with Telchemy, and are thus
# subject to the terms of that license agreement or confidentiality agreement
# and any other applicable agreement between the party and Telchemy.  If
# there is any doubt as to whether a party is entitled to access, read, use,
# copy, adapt, modify or otherwise utilize the Software, or whether a party
# is entitled to disclose the Software to any other party, you should contact
# Telchemy.  If you, as a party, have not entered into a license agreement or
# confidentiality agreement with Telchemy granting access to this Software,
# all media, copies and printed listings containing the Software should be
# forthwith returned to Telchemy.
#
# Telchemy reserves the right to take legal action against any party that
# violates Telchemy's rights to the Software, including without limitation a
# party's breach of the above conditions.
#
# If you have questions about any of the above conditions, or wish to report
# violations, please contact:  support@telchemy.com
#
# ##########################################################################*/

/* tcpstrm.c
 *
 * VQmon Sample Host Application - TCP Streaming
 *
 * Copyright (c) 2011-2012 Telchemy, Incorporated
 */

#include <stdio.h>
#include <stdlib.h>

#include "tcpstrm.h"

#include "vqmon.h"

/*
 * This application demonstrates the usage of VQmon in monitoring an TCP
 * streaming session. Media data is read from a file and indicated to VQmon
 * using VQmonInterfaceIndicatePacket, and VQmon uses the stream management
 * callback set in VQmonInitialize to indicate stream-related events to the
 * application.
 */

int main(int argc, char* argv[])
{
    FILE* _hFile;
    vqmon_initparams_t _tInitParams;
    vqmon_handle_t _hInterface;
    vqmon_time_t _tTimestamp;
    vqmon_pktdesc_t _tPktDesc;
    vqmon_result_t _result;

    vqmon_interfacestats_pkt_t _tPktStats;
    tcmyU32 _nSize;
    tcmyU32 _nSeqNum = 0;
    tcmyU16 _nTCPHdrOffset = 20;
    static tcmyU8 _tPktBuf[1536];
    tcmyU32 _nDataLength;
    tcmyU32 _nRead;
    tcmyCHAR _aHTTPHdr[512];
    tcmyCHAR _aContentType[64];


    if (argc == 2)
    {
        _hFile = fopen(argv[1], "rb");

        if (NULL == _hFile)
        {
            fprintf(stderr, "cannot open file %s!\n", argv[1]);
            exit(EXIT_FAILURE);
        }
    }
    else
    {
        printf("usage: %s filename\n", argv[0]);
        printf("\t- a media file with extension .ts, .mp4 or .flv\n");
        exit(EXIT_FAILURE);
    }

    _nSize = VQMON_STRLEN(argv[1]);
    if (!VQMON_STRCMP((argv[1] + _nSize - 4), ".flv"))
    {
        VQMON_STRCPY(_aContentType, "Content-Type: video/x-flv\r\n");
    }
    else if (!VQMON_STRCMP((argv[1] + _nSize - 3), ".ts"))
    {
        VQMON_STRCPY(_aContentType, "Content-Type: video/mp2t\r\n");
    }
    else if (!VQMON_STRCMP((argv[1] + _nSize - 4), ".mp4"))
    {
        VQMON_STRCPY(_aContentType, "Content-Type: video/mp4\r\n");
    }
    else
    {
            fprintf(stderr, "invalid media file %s!\n", argv[1]);
            exit(EXIT_FAILURE);
    }

    /*
     * Initialize the VQmon Library
     */
    VQMON_MEMCLEAR(&_tInitParams, sizeof(vqmon_initparams_t));
    _tInitParams.nVersion = VQMON_IFVERSION_CURRENT;
    _tInitParams.pfStreamNotifyHdlr = TCPStrm_NotifyHandler;
    _result = VQmonInitialize(&_tInitParams);
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonInitialize failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    VQmonDebugSetTraceLevel(3);

    /*
     * Create an interface handle
     */
    _result = VQmonInterfaceCreate(&_hInterface, NULL);
    if (VQMON_ESUCCESS != _result)
    {
        fprintf(stderr, "VQmonInterfaceCreate failed with code %d\n", _result);
        exit(EXIT_FAILURE);
    }

    /*
     * Get the size of the media file.
     */
    fseek(_hFile, 0L, SEEK_END);
    _nSize = ftell(_hFile);
    rewind(_hFile);

    _tTimestamp.tv_sec = 0;
    _tTimestamp.tv_usec = 0;

    /*
     * Set the basic packet descriptor information.
     */
    VQMON_MEMCLEAR(&_tPktDesc, sizeof(vqmon_pktdesc_t));
    _tPktDesc.ePktStartIndicator = vqmonPktDescStartTCPHeader;
    _tPktDesc.nCapturedPktLen = sizeof(_tPktBuf);
    _tPktDesc.nTotalPktLen = sizeof(_tPktBuf);
    _tPktDesc.pPktBuffer = &_tPktBuf[0];

    _tPktDesc.ipinfo.tSrcAddress.eVersion = vqmonAddressIPv4;
    _tPktDesc.ipinfo.nProtocolId = 6;

    _tPktDesc.ipinfo.tSrcAddress.ip.v4.octet[0] = 192;
    _tPktDesc.ipinfo.tSrcAddress.ip.v4.octet[0] = 168;
    _tPktDesc.ipinfo.tSrcAddress.ip.v4.octet[0] = 1;
    _tPktDesc.ipinfo.tSrcAddress.ip.v4.octet[0] = 100;
    _tPktDesc.ipinfo.tDstAddress.eVersion = vqmonAddressIPv4;

    _tPktDesc.ipinfo.tDstAddress.ip.v4.octet[0] = 192;
    _tPktDesc.ipinfo.tDstAddress.ip.v4.octet[0] = 168;
    _tPktDesc.ipinfo.tDstAddress.ip.v4.octet[0] = 2;
    _tPktDesc.ipinfo.tDstAddress.ip.v4.octet[0] = 200;
    _tPktDesc.ipinfo.fHdrFlags |= VQMON_IPINFO_VALID;

    VQMON_MEMCLEAR(_tPktBuf, sizeof(_tPktBuf));
    /* TCP source port */
    _tPktBuf[1] = 80;
    /* TCP destination port */
    _tPktBuf[2] = 16;
    /* TCP data offset (header length) */
    _tPktBuf[12] = (_nTCPHdrOffset << 2);

    /*
     * Create a TCP SYNC-ACK packet during TCP connection setup (handshake).
     * Note: Not really necessary for this application.
     */
    _tPktDesc.nCapturedPktLen = _tPktDesc.nTotalPktLen =

    _tPktDesc.nCurrPDULen = _tPktDesc.nCurrPktBufferLen = _nTCPHdrOffset;
    /* TCP header six bit flags */
    _tPktBuf[13] = 0x12;    /* SYNC-ACK */

    /*
     * Indicate the packet to VQmon.
     */
    VQmonInterfaceIndicatePacket(
    _hInterface,
    _tTimestamp,
    &_tPktDesc
    );

    /*
     * Update the sequence number (needed).
     */
    _nSeqNum++;

    /*
     * Create a TCP packet with HTTP header information.
     */
    VQMON_STRCPY(_aHTTPHdr, "HTTP/1.1 200 OK\r\n");
    VQMON_STRCAT(_aHTTPHdr, "Accept-Ranges: bytes\r\n");
    VQMON_STRCAT(_aHTTPHdr, _aContentType);

    _nDataLength = VQMON_STRLEN(_aHTTPHdr);
    sprintf(&_aHTTPHdr[_nDataLength], "Content-Length: %d\r\n\r\n", _nSize);
    _nDataLength = VQMON_STRLEN(_aHTTPHdr);

    /*
     * Output the HTTP header information.
     */
    printf("%s", _aHTTPHdr);

    VQMON_MEMCOPY(&_tPktBuf[_nTCPHdrOffset], _aHTTPHdr, _nDataLength);

    _tPktDesc.ePktStartIndicator = vqmonPktDescStartTCPHeader;
    _tPktDesc.nCapturedPktLen = _tPktDesc.nTotalPktLen =

    _tPktDesc.nCurrPDULen = _tPktDesc.nCurrPktBufferLen = _nTCPHdrOffset + _nDataLength;
    _tPktDesc.pPktBuffer = &_tPktBuf[0];

    _tPktBuf[4] = (_nSeqNum >> 24) & 0xFF;
    _tPktBuf[5] = (_nSeqNum >> 16) & 0xFF;
    _tPktBuf[6] = (_nSeqNum >> 8 ) & 0xFF;
    _tPktBuf[7] = _nSeqNum & 0xFF;

    /* TCP header six bit flags */
    _tPktBuf[13] = 0;

    _tTimestamp.tv_usec += 1875;

    /*
     * Indicate the packet to VQmon.
     */
    VQmonInterfaceIndicatePacket(
    _hInterface,
    _tTimestamp,
    &_tPktDesc
    );

    /*
     * Update the sequence number.
     */
    _nSeqNum += _nDataLength;
    _nDataLength = 188;

    /*
     * Continuously read data from the file, create TCP packet and feed it to VQmon.
     */
   while ( !feof(_hFile) )
   {
        VQMON_MEMCLEAR(_tPktBuf, sizeof(_tPktBuf));

        /*
         * Read some data from the media file.
         */
        _nRead = fread(&_tPktBuf[_nTCPHdrOffset], 1, _nDataLength, _hFile);

        /*
         * Set the basic packet descriptor information.
          */
        VQMON_MEMCLEAR(&_tPktDesc, sizeof(vqmon_pktdesc_t));
        _tPktDesc.ePktStartIndicator = vqmonPktDescStartTCPHeader;
        _tPktDesc.nCapturedPktLen = sizeof(_tPktBuf);
        _tPktDesc.nTotalPktLen = sizeof(_tPktBuf);
        _tPktDesc.pPktBuffer = &_tPktBuf[0];

        _tPktDesc.ipinfo.tSrcAddress.eVersion = vqmonAddressIPv4;
        _tPktDesc.ipinfo.nProtocolId = 6;

        _tPktDesc.ipinfo.tSrcAddress.ip.v4.octet[0] = 192;
        _tPktDesc.ipinfo.tSrcAddress.ip.v4.octet[0] = 168;
        _tPktDesc.ipinfo.tSrcAddress.ip.v4.octet[0] = 1;
        _tPktDesc.ipinfo.tSrcAddress.ip.v4.octet[0] = 100;
        _tPktDesc.ipinfo.tDstAddress.eVersion = vqmonAddressIPv4;

        _tPktDesc.ipinfo.tDstAddress.ip.v4.octet[0] = 192;
        _tPktDesc.ipinfo.tDstAddress.ip.v4.octet[0] = 168;
        _tPktDesc.ipinfo.tDstAddress.ip.v4.octet[0] = 2;
        _tPktDesc.ipinfo.tDstAddress.ip.v4.octet[0] = 200;
        _tPktDesc.ipinfo.fHdrFlags |= VQMON_IPINFO_VALID;

        /* TCP source port */
        _tPktBuf[1] = 80;
        /* TCP destination port */
        _tPktBuf[2] = 16;
        /* TCP data offset (header length) */
        _tPktBuf[12] = (_nTCPHdrOffset << 2);

        /*
         * Update the packet descriptor information.
         */

        _tPktDesc.nCapturedPktLen = _tPktDesc.nTotalPktLen =
            _tPktDesc.nCurrPDULen = _tPktDesc.nCurrPktBufferLen = _nTCPHdrOffset + _nRead;

        _tPktBuf[4] = (_nSeqNum >> 24) & 0xFF;
        _tPktBuf[5] = (_nSeqNum >> 16) & 0xFF;
        _tPktBuf[6] = (_nSeqNum >> 8 ) & 0xFF;
        _tPktBuf[7] = _nSeqNum & 0xFF;

        /*
         * Update the local packet receive time stamp.
         */

        _tTimestamp.tv_usec += 1875;

        if (_tTimestamp.tv_usec >= 1000000)
        {
            _tTimestamp.tv_sec++;
            _tTimestamp.tv_usec -= 1000000;
        }


        VQmonInterfaceIndicatePacket(
        _hInterface,
        _tTimestamp,
        &_tPktDesc
        );

        /*
         * Update the sequence number.
         */

        _nSeqNum += _nRead;
   }

    /*
     * Terminate all streams.
     * This will produce vqmonStreamMgmtNotifyTerminated callbacks, and
     * TCPStrm_NotifyHandler will display metrics associated with the
     * stream and destroy the stream handle.
     */
    printf("\n--- Terminating streams\n");
    _result = VQmonInterfaceStreamsCleanupInactive(
            _hInterface,
            _tTimestamp,
            0
            );

    /*
     * Retrieve and display aggregate statistics for the interface
     */
    _nSize = sizeof(vqmon_interfacestats_pkt_t);
    VQMON_MEMCLEAR(&_tPktStats, _nSize);
    _result = VQmonInterfaceStatisticsGet(
            _hInterface,
            VQMON_IFMETRICBLOCKID_PACKETSTATS,
            &_nSize,
            &_tPktStats
            );

    printf("\n%d packets received on interface\n\n", (int) _tPktStats.nPktsRcvd);

    /*
     * Cleanup
     */
    VQmonInterfaceDestroy(_hInterface);
    VQmonCleanup();

    fclose(_hFile);

    return 0;
}

vqmon_result_t
TCPStrm_NotifyHandler(
    const vqmon_handle_t hInterface,
    const vqmon_handle_t hStream,
    const vqmon_streammgmtnotifytype_t eNotification,
    vqmon_streammgmtnotify_info_t *pNotifyInfo
    )
{
    switch (eNotification)
    {
        case vqmonStreamMgmtNotifyCreated:
        {
            printf("TCPStrm_NotifyHandler: Stream created, handle %p\n",
                    hStream);
            break;
        }

        case vqmonStreamMgmtNotifyActivating:
        {
            printf("TCPStrm_NotifyHandler: Stream activating, handle %p\n",
                    hStream);
            break;
        }

        case vqmonStreamMgmtNotifyTerminated:
        {
            /* Display metrics for the stream */
            switch (pNotifyInfo->eStreamType)
            {
                case vqmonStreamTypeVideo:
                {
                    TCPStrm_VideoSummaryDisplay(hStream);
                    break;
                }

                case vqmonStreamTypeAudio:
                {
                    TCPStrm_AudioSummaryDisplay(hStream);
                    break;
                }

                default:
                {
                    printf("TCPStrm_NotifyHandler: Unsupported stream type terminated: %p\n",
                            hStream);
                    break;
                }
            }

            /* Free the memory used by the stream */
            VQmonStreamDestroy(hStream);
            break;
        }

        default:
        {
            break;
        }
    }
    return VQMON_ESUCCESS;
}

void
TCPStrm_VideoSummaryDisplay(
    vqmon_handle_t const hStream
    )
{
    tcmyU32 _nSize;
    vqmon_result_t _result;

    vqmon_streamcfg_transprotoext_t _tTransProto;
    vqmon_streammetrics_pkttrans_t _tTransMetrics;
    vqmon_streammetrics_videoqual_t _tVideoQual;

    _nSize = sizeof(vqmon_streamcfg_transprotoext_t);
    VQMON_MEMCLEAR(&_tTransProto, _nSize);
    _tTransProto.eTransportProto = vqmonStreamTransProtoMP2TS;
    _result = VQmonStreamConfigGet(
            hStream,
            VQMON_STREAMCFGBLOCKID_TRANSPORTINFO,
            &_nSize,
            &_tTransProto
            );

    _nSize = sizeof(vqmon_streammetrics_pkttrans_t);
    VQMON_MEMCLEAR(&_tTransMetrics, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_PKTTRANS,
            &_nSize,
            &_tTransMetrics
            );

    _nSize = sizeof(vqmon_streammetrics_videoqual_t);
    VQMON_MEMCLEAR(&_tVideoQual, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_VIDEOQUAL,
            &_nSize,
            &_tVideoQual
            );

    printf("Video stream %p PID %d terminating, %d packets received\n", 
            hStream,
            _tTransProto.proto.mp2ts.nPID,
            _tTransMetrics.nPktsRcvd);
    printf("Rel. MOS-V (min/avg/max): %3.2f/%3.2f/%3.2f\n",
            _tVideoQual.nMinRelativeMOS_V / 256.0,
            _tVideoQual.nAvgRelativeMOS_V / 256.0,
            _tVideoQual.nMaxRelativeMOS_V / 256.0
          );
}

void
TCPStrm_AudioSummaryDisplay(
    vqmon_handle_t const hStream
    )
{
    tcmyU32 _nSize;
    vqmon_result_t _result;

    vqmon_streamcfg_transprotoext_t _tTransProto;
    vqmon_streammetrics_pkttrans_t _tTransMetrics;
    vqmon_streammetrics_audioqual_t _tAudioQual;

    _nSize = sizeof(_tTransProto);
    VQMON_MEMCLEAR(&_tTransProto, _nSize);
    _tTransProto.eTransportProto = vqmonStreamTransProtoMP2TS;
    _result = VQmonStreamConfigGet(
            hStream,
            VQMON_STREAMCFGBLOCKID_TRANSPORTINFO,
            &_nSize,
            &_tTransProto
            );

    _nSize = sizeof(_tTransMetrics);
    VQMON_MEMCLEAR(&_tTransMetrics, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_PKTTRANS,
            &_nSize,
            &_tTransMetrics
            );

    _nSize = sizeof(_tAudioQual);
    VQMON_MEMCLEAR(&_tAudioQual, _nSize);
    _result = VQmonStreamMetricsGet(
            hStream,
            VQMON_STREAMMETRICBLOCKID_AUDIOQUAL,
            &_nSize,
            &_tAudioQual
            );

    printf("Audio stream %p PID %d terminating, %d packets received\n", 
            hStream,
            _tTransProto.proto.mp2ts.nPID,
            _tTransMetrics.nPktsRcvd);
    printf("MOS-A (min/avg/max): %3.2f/%3.2f/%3.2f\n",
            _tAudioQual.nMinMOS_A / 256.0,
            _tAudioQual.nAvgMOS_A / 256.0,
            _tAudioQual.nMaxMOS_A / 256.0
          );
}

